import { constructMetadata } from "@/components/utils/seo";
import fs from 'fs';
import path from 'path';
import { notFound } from "next/navigation";

interface LocalizedText {
  ar: string;
  en: string;
}

interface PageData {
  seo: {
    title: LocalizedText;
    description: LocalizedText;
  };
}

const getPageData = async (contentSlug: string): Promise<PageData | null> => {
  try {
    const filePath = path.join(process.cwd(), 'public', 'data', `${contentSlug}.json`);
    
    if (!fs.existsSync(filePath)) return null;

    const jsonData = fs.readFileSync(filePath, 'utf8');
    return JSON.parse(jsonData);
  } catch (error) {
    console.error('Error reading data:', error);
    return null;
  }
}

type Props = {
  params: Promise<{ locale: string; content: string }>;
};

export async function generateMetadata({ params }: Props) {
  const { locale, content: slug } = await params;
  const data = await getPageData(slug);

  if (!data) return {};

  const currentLocale = (locale as 'ar' | 'en') || "ar";

  return constructMetadata({
    title: data.seo.title[currentLocale],
    description: data.seo.description[currentLocale],
    canonicalUrlRelative: `/${locale}/pages/${slug}`,
  });
}

export default async function Page({ params }: Props) {
  const { locale, content: slug } = await params;
  const data = await getPageData(slug);

  if (!data) {
    notFound();
  }

  return (
    <></>
  );
}